'
' SAP Inspection Scope Validation Script
' Author: Dario Pascoal
'
' Description: This VBScript automates the process of checking whether a specific 
' material/tool is within scope for inspection procedures within the SAP ZPACK 
' transaction system. The script serves as a critical decision point in the DHL 
' inspection workflow, determining whether tools should proceed to detailed 
' inspection or be returned to field service operations.
'
' Business Purpose:
' In DHL's quality management process, not all tools require detailed inspection.
' This script automates the scope determination by reading specific text fields
' in SAP that indicate inspection requirements. Based on the scope status, the
' script directs the workflow to either continue with inspection details or
' return the tool to field service for alternative processing.
'
' Workflow Integration:
' 1. Receives material number from calling application
' 2. Connects to active SAP GUI session
' 3. Navigates to ZPACK transaction for material lookup
' 4. Searches for specified material number
' 5. Reads inspection scope text field content
' 6. Analyzes scope text against predefined patterns
' 7. Returns appropriate status code for workflow routing
' 8. Performs SAP navigation cleanup before exit
'
' Technical Implementation:
' - VBScript with COM object automation for SAP GUI interaction
' - Pattern matching for scope text analysis and classification
' - Command-line interface with material number and logging parameters
' - Comprehensive error handling with graceful degradation
' - File-based logging system for audit trails and debugging
' - Registry security configuration for unattended automation
'
' Integration Points:
' - SAP GUI Scripting Engine for automated navigation and data extraction
' - DHL All-in-One main application for status reporting and workflow control
' - Windows Shell for security dialog handling and registry management
' - File System Object for logging operations and error tracking
' - ZPACK transaction for material data access and scope determination
'
' Security Features:
' - Temporary registry modifications with automatic cleanup procedures
' - Error handling to prevent script crashes and data corruption
' - Detailed logging for security audit trails and compliance
' - Secure SAP GUI connection handling with proper authentication
' - Automatic cleanup procedures to prevent resource leaks
'
' Return Codes:
' - OUT-OF-SCOPE-RETURN-TO-FIELD-SERVICE: Tool requires field service processing
' - IN-SCOPE-PROCEED-TO-INSPECTION: Tool continues to inspection details
' - OTHER-TEXT-PROCEED-TO-INSPECTION: Default case for undefined scope text
' - ERROR-FIELD-NOT-FOUND: Critical error when scope field is inaccessible
'

' =============================================================================
' GLOBAL ERROR HANDLING CONFIGURATION
' =============================================================================
' Enable comprehensive error handling to prevent script crashes and provide
' graceful degradation when encountering unexpected SAP GUI conditions
On Error Resume Next

' =============================================================================
' ENVIRONMENT INITIALIZATION
' =============================================================================
' Initialize Windows Script Host shell object for system interactions
' This object handles registry modifications and security dialog automation
Set WSHShell = CreateObject("WScript.Shell")

' =============================================================================
' SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Disable SAP GUI scripting security warnings to enable automated interaction
' These registry modifications are essential for unattended script execution
' and prevent user prompts that would interrupt the automation workflow

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt automation processes
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Disable allow list requirement warnings for automated scripting
' This prevents security prompts that would require manual user interaction
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"

' Disable allow list usage to prevent script blocking
' This allows our automation script to execute without whitelist restrictions
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' =============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' =============================================================================
' Extract and validate command line arguments for material number and logging
' The script requires a material number and optionally accepts a log file path
' for detailed operation tracking and troubleshooting capabilities

' Declare log file objects and path variables
' These objects handle all file-based logging operations
Dim logFile, logFilePath

' Get material number from first command line argument (required)
' This material number identifies the equipment for scope validation
materialNumber = WScript.Arguments.Item(0)

' =============================================================================
' LOGGING SYSTEM INITIALIZATION
' =============================================================================
' Setup comprehensive logging system for operation tracking and debugging
' The logging system supports detailed file logging with timestamps for
' audit trails, troubleshooting, and operational monitoring

' Initialize logging system if log file path is provided as second argument
' Enhanced logging provides detailed operation tracking for debugging
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    
    ' Attempt to open log file in append mode for continuous logging
    ' Append mode (8) preserves existing log entries and adds new ones
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle log file opening errors gracefully
    ' If logging fails, continue script execution without logging
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script startup information for audit trail
        ' This creates a clear session marker in the log file
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "========================================="
    End If
End If

' =============================================================================
' LOGGING UTILITY FUNCTIONS
' =============================================================================
' Centralized logging functions to ensure consistent operation tracking
' and error reporting across all script operations and SAP interactions

' File-only logging function for detailed internal operations
' Used for technical details that don't need console output
Sub LogToFile(message)
    On Error Resume Next
    ' Only write to file if logging is enabled and file is available
    If Not logFile Is Nothing Then
        ' Format: timestamp - [script name] - message for easy parsing
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Dual logging function for important messages and user feedback
' Outputs to both console (for immediate feedback) and log file (for records)
Sub LogMessage(message)
    ' Display message to user for immediate feedback
    WScript.Echo message
    ' Also save to log file for permanent record
    LogToFile message
End Sub

' =============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' Establish connection to SAP GUI application and active session
' This section handles the complex process of connecting to SAP through
' the SAP GUI Scripting Engine and manages connection error handling

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto = GetObject("SAPGUI")
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' Children(0) accesses the first active connection to SAP
   Set connection = application.Children(0)
End If

' Initialize SAP session object for user interaction simulation
' The session object provides access to SAP screens and transactions
If Not IsObject(session) Then
   ' Get the first (primary) session from the active connection
   ' This session will be used for all SAP GUI automation commands
   Set session = connection.Children(0)
End If

' Connect VBScript event handlers to SAP GUI objects
' This enables the script to respond to SAP GUI events and errors
If IsObject(WScript) Then
   ' Connect session events for user interaction monitoring
   ' "on" prefix enables event handling for session-level events
   WScript.ConnectObject session, "on"
   ' Connect application events for system-level monitoring
   ' This handles application-wide events and error conditions
   WScript.ConnectObject application, "on"
End If

' =============================================================================
' SECURITY DIALOG AUTOMATION
' =============================================================================
' Automated handling of SAP GUI security dialogs that may appear during
' script execution, ensuring unattended operation without user intervention

'
' Handle security dialog automation for unattended script execution
' This function automatically dismisses security warnings that could
' interrupt the automation workflow and require manual intervention
'
' The function waits up to 10 seconds for security dialogs to appear,
' then automatically accepts them to allow script continuation. This is
' essential for unattended operation in production environments.
'
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Wait up to 10 seconds for security dialog to appear
    ' This prevents infinite waiting while allowing time for dialog display
    Do While timeout < 10
        ' Check if SAP GUI Security dialog is active
        ' AppActivate returns True if the specified window title is found
        If shell.AppActivate("SAP GUI Security") Then
            ' Brief pause to ensure dialog is fully loaded
            ' This prevents sending keys before dialog is ready
            WScript.Sleep 500
            
            ' Press space to check "Remember my decision" checkbox
            ' This prevents the same security dialog from appearing repeatedly
            shell.SendKeys " "
            
            ' Brief pause between key operations for reliability
            ' Ensures the checkbox state change is processed
            WScript.Sleep 200
            
            ' Press Enter to click the "Allow" button
            ' This grants permission for the script to continue execution
            shell.SendKeys "{ENTER}"
            dialogFound = True
            ' Exit loop immediately after successful dialog handling
            Exit Do
        End If
        
        ' Wait 500ms before checking again to avoid excessive CPU usage
        ' This provides a reasonable balance between responsiveness and performance
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' =============================================================================
' USER INTERFACE COMPATIBILITY FUNCTIONS
' =============================================================================
' Legacy compatibility functions for UI integration and popup handling
' These functions maintain compatibility with older versions of the system

'
' Legacy popup function for out-of-scope notifications
' This function is maintained for backward compatibility but no longer
' displays popups since UI handling has been moved to JavaScript layer
' 
' Modern Implementation Note:
' Popup notifications are now handled in the JavaScript UI layer for
' better user experience and consistent styling with the application
'
Sub ShowOutOfScopePopup()
    ' Popup functionality has been migrated to JavaScript UI layer
    ' This function is retained for compatibility with legacy code
    ' but performs no operations since popup handling is now client-side
End Sub

' =============================================================================
' MAIN SCOPE VALIDATION WORKFLOW
' =============================================================================
' Core business logic for validating material inspection scope within SAP
' This function orchestrates the complete scope checking process including
' SAP navigation, data extraction, analysis, and workflow routing decisions

'
' Execute comprehensive inspection scope validation for specified material
' 
' This function performs the complete scope checking workflow:
' 1. Connects to SAP and navigates to ZPACK transaction
' 2. Searches for the specified material number
' 3. Extracts and analyzes scope determination text
' 4. Makes routing decisions based on scope analysis
' 5. Returns appropriate status codes for workflow continuation
' 
' Business Logic:
' - OUT OF SCOPE: Returns tool to field service for alternative processing
' - IN SCOPE: Continues to detailed inspection procedures
' - OTHER TEXT: Defaults to inspection continuation for safety
' 
' Error Handling:
' - Graceful handling of missing fields and SAP connection issues
' - Comprehensive logging for debugging and audit purposes
' - Automatic SAP navigation cleanup on all exit paths
'
Sub CheckInspectionScope()
    LogMessage "DEBUG: Starting scope check for material: " & materialNumber
    
    ' Execute security dialog handling before starting SAP operations
    ' This ensures any security prompts are automatically dismissed
    HandleSecurityDialog()
    
    ' ==========================================================================
    ' SAP TRANSACTION NAVIGATION
    ' ==========================================================================
    ' Navigate to ZPACK transaction for material scope determination
    ' ZPACK provides access to material-specific inspection requirements
    
    ' Maximize SAP GUI window to ensure all elements are visible
    ' This prevents issues with hidden or partially visible UI elements
    session.findById("wnd[0]").maximize
    
    ' Navigate to ZPACK transaction using the command field
    ' The transaction code directs SAP to the material packaging interface
    session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
    
    ' Execute the transaction command (equivalent to pressing Enter)
    ' sendVKey 0 sends the Enter key to execute the transaction
    session.findById("wnd[0]").sendVKey 0
    
    ' ==========================================================================
    ' MATERIAL SEARCH EXECUTION
    ' ==========================================================================
    ' Input material number and execute search to locate scope information
    ' This section handles the material search process within ZPACK transaction
    
    ' Enter the material number in the material search field
    ' S_MATNR-LOW is the standard SAP field for material number input
    session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber
    
    ' Execute the search using the Execute button (Function 8)
    ' This triggers the material lookup and displays scope information
    session.findById("wnd[0]/tbar[1]/btn[8]").press
    
    ' ==========================================================================
    ' SCOPE TEXT FIELD EXTRACTION
    ' ==========================================================================
    ' Locate and extract the inspection scope determination text field
    ' This field contains the critical information for workflow routing decisions
    
    ' Attempt to locate the scope text field with error handling
    ' The OPERCOM field contains inspection scope determination text
    On Error Resume Next
    Set textField = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/txtWA_OUTPUT_9100-OPERCOM")
    
    ' Handle field location errors gracefully
    ' If the field cannot be found, perform cleanup and exit with error status
    If Err.Number <> 0 Then
        LogMessage "ERROR: Could not find text field: " & Err.Description
        
        ' Perform SAP navigation cleanup before error exit
        ' This ensures SAP returns to a clean state for subsequent operations
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        ' Return error status with descriptive message
        WScript.Echo "ERROR-FIELD-NOT-FOUND|Field not found"
        Exit Sub
    End If
    
    ' Focus on the text field to ensure reliable data extraction
    ' This step ensures the field is active and readable
    textField.setFocus
    
    ' Brief pause to allow field focus and data loading
    ' This prevents reading empty or incomplete field data
    WScript.Sleep 1000
    
    ' ==========================================================================
    ' SCOPE TEXT ANALYSIS AND PROCESSING
    ' ==========================================================================
    ' Extract, validate, and analyze the scope determination text for workflow
    ' routing decisions based on predefined business rule patterns
    
    ' Declare variables for text processing and analysis
    ' These variables handle both raw and processed text content
    Dim scopeText, originalText
    originalText = ""
    
    ' Extract text content from the scope field with error handling
    ' This ensures we capture the actual field content reliably
    On Error Resume Next
    originalText = textField.Text
    
    ' Handle text extraction errors and ensure we have valid string data
    If Err.Number <> 0 Then
        originalText = ""
        Err.Clear
    End If
    On Error Goto 0
    
    ' Normalize text data to handle null and empty values consistently
    ' This prevents comparison errors with undefined or null data
    If IsNull(originalText) Then originalText = ""
    If originalText = Empty Then originalText = ""
    
    ' Convert to uppercase and trim whitespace for reliable pattern matching
    ' This standardizes the text for consistent comparison logic
    scopeText = Trim(UCase(originalText))
    
    ' ==========================================================================
    ' DEBUGGING AND DIAGNOSTIC LOGGING
    ' ==========================================================================
    ' Comprehensive logging for troubleshooting and audit trail purposes
    ' This section provides detailed information about the analysis process
    
    LogMessage "DEBUG: Found scope text: '" & scopeText & "'"
    LogMessage "DEBUG: Raw text field content: '" & originalText & "'"
    LogMessage "DEBUG: Text field length: " & Len(originalText)
    LogMessage "DEBUG: Looking for 'OUT OF SCOPE FOR INSPECTION' pattern..."
    LogMessage "DEBUG: Looking for 'OUT OF SCOPE' pattern..."
    LogMessage "DEBUG: Looking for 'IN SCOPE FOR INSPECTION' pattern..."
    
    ' Detailed pattern matching diagnostics for debugging
    ' These logs help troubleshoot pattern matching issues
    LogMessage "DEBUG: InStr result for 'OUT OF SCOPE FOR INSPECTION': " & InStr(scopeText, "OUT OF SCOPE FOR INSPECTION")
    LogMessage "DEBUG: InStr result for 'OUT OF SCOPE': " & InStr(scopeText, "OUT OF SCOPE")
    LogMessage "DEBUG: InStr result for 'IN SCOPE FOR INSPECTION': " & InStr(scopeText, "IN SCOPE FOR INSPECTION")
    
    ' ==========================================================================
    ' SCOPE DETERMINATION AND WORKFLOW ROUTING
    ' ==========================================================================
    ' Analyze scope text content and make workflow routing decisions based on
    ' business rules and inspection requirements for the specified material
    
    ' Primary check: Material is explicitly out of scope for inspection
    ' This is the highest priority condition that routes tools to field service
    If InStr(scopeText, "OUT OF SCOPE FOR INSPECTION") > 0 Or InStr(scopeText, "OUT OF SCOPE") > 0 Then
        LogMessage "INFO: Tool is out of scope for inspection"
        
        ' Perform SAP navigation cleanup before routing decision
        ' This ensures SAP returns to proper state for subsequent operations
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning OUT-OF-SCOPE with text: '" & originalText & "'"
        
        ' Return out-of-scope status to route tool back to field service
        ' Include original text content for audit trail and user information
        WScript.Echo "OUT-OF-SCOPE-RETURN-TO-FIELD-SERVICE|" & originalText
        
    ' Secondary check: Material is explicitly in scope for inspection  
    ' This condition continues the workflow to detailed inspection procedures
    ElseIf InStr(scopeText, "IN SCOPE FOR INSPECTION") > 0 Then
        LogMessage "INFO: Tool is in scope for inspection - proceeding to inspection details"
        
        ' Perform SAP navigation cleanup before continuing workflow
        ' This prepares SAP for the next phase of inspection processing
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning IN-SCOPE with text: '" & originalText & "'"
        
        ' Return in-scope status to continue to inspection details
        ' Include original text content for comprehensive audit logging
        WScript.Echo "IN-SCOPE-PROCEED-TO-INSPECTION|" & originalText
        
    ' Default case: Scope text is undefined or contains other content
    ' Safety default routes to inspection to avoid missing required inspections
    Else
        LogMessage "INFO: Scope text is something else ('" & scopeText & "') - proceeding to inspection details"
        
        ' Perform standard SAP navigation cleanup
        ' This maintains consistent SAP state regardless of decision path
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning OTHER-TEXT with text: '" & originalText & "'"
        
        ' Return default status to proceed with inspection (safe approach)
        ' Include original text content for analysis and potential rule updates
        WScript.Echo "OTHER-TEXT-PROCEED-TO-INSPECTION|" & originalText
    End If
End Sub

' =============================================================================
' SCRIPT EXECUTION AND WORKFLOW ORCHESTRATION
' =============================================================================
' Main script execution point that orchestrates the complete scope validation
' workflow and ensures proper resource cleanup upon completion

' Execute the primary scope validation workflow
' This calls the main business logic function to perform scope checking
CheckInspectionScope()

' =============================================================================
' CLEANUP AND FINALIZATION PROCEDURES
' =============================================================================
' Comprehensive cleanup procedures to ensure proper resource management
' and logging finalization before script termination

'
' Cleanup function to properly close log file and release system resources
' 
' This function ensures proper log file closure and resource cleanup to
' prevent memory leaks and ensure all log data is properly written to disk.
' The function handles errors gracefully to ensure cleanup always completes.
'
Sub CloseLogFile()
    On Error Resume Next
    ' Check if log file is still open and available for closure
    If Not logFile Is Nothing Then
        ' Write session end marker to log for clear audit trail separation
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        
        ' Properly close the log file to ensure all buffered data is written
        logFile.Close
        
        ' Release the log file object to free system resources
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup procedures before script termination
' This ensures proper resource cleanup and log file closure in all cases
CloseLogFile()
